/***********************************************************************
 
NAME
	shortshuf(), intshuf(), floatshuf(), doubleshuf()
	shortshuf_(), intshuf_(), floatshuf_(), doubleshuf_()

SYNOPSIS
	C interface:
		shortshuf(num)
		short *num;
 
		intshuf(num)
		int *num;

		floatshuf(num)
		float *num;

		doubleshuf(num)
		double *num;

	FORTRAN77 interface:
		integer*4 i, shortshuf
		integer*2 num
		i = shortshuf(num)
 
		integer*2 i, intshuf
		integer*4 num
		i = intshuf(num)

		integer*4 i, floatshuf
		real*4 num
		i = floatshuf(num)

		integer*4 i, doubleshuf
		real*8 num
		i = doubleshuf(num)

DESCRIPTION
	These functions convert data types from VAX VMS integer
	and floating point formats to SUN UNIX (IEEE) formats.
 
	The converted number is returned as the argument.
	The return value of the function is undefined.

*************************************************************************/
 
#include	<stdio.h>
 
/* shortshuf converts VAX 16 bit short integers to Sun shorts */

shortshuf(num)
short *num;
{
	short left, right;
	
	right = left = *num;
	right >>= 8;
	right &= 0377;
	left <<= 8;
	*num = left|right;
}	

/* intshuf changes VAX 32 bit integers to Sun integers */

intshuf(num)
int *num;
{
	int b1,b2,b3,b4;
	
	b1 = b2 = b3 = b4 = *num;
	b1 >>= 24;
	b2 <<= 8;
	b3 >>= 8;
	b4 <<= 24;

	b2 &= 077600000;
	b3 &= 0177400;
	b1 &= 0377;

	*num = b1|b2|b3|b4;
}	

/* floatshuf changes VAX 32 bit floats to Sun floats */

floatshuf(num)
union {
	float r4;
	unsigned ui4;
} *num;

{
	unsigned b1,b2,b3,b4;

	b1 = b2 = b3 = b4 = num->ui4;
	b1 >>= 8;
	b2 <<= 8;
	b3 >>= 8;
	b4 <<= 8;

	b1 &= 077600000;
	b2 &= 037700000000;
	if (b2 != 0) {
		if (b2 >= 2) {
			b2 >>= 23;
			b2 -= 2;
			b2 <<= 23;
		} else {
			fprintf(stderr,"\nWarning: real*4 number too small");
		}
	}
	b3 &= 0377;
	b4 &= 0177400;

	num->ui4 = (b1|b2|b3|b4);
}	

/* doubleshuf changes VAX 64 bit doubles to Sun doubles */

doubleshuf(num)
union {
	double   r8;
	unsigned ui4[2];
	char     buf[8];
} *num;

{
	char	temp[8];
	unsigned u0, u1, u2, u3, u4;
	int i;

	temp[0] = num->buf[1];
	temp[1] = num->buf[0];
	temp[2] = num->buf[3];
	temp[3] = num->buf[2];
	temp[4] = num->buf[5];
	temp[5] = num->buf[4];
	temp[6] = num->buf[7];
	temp[7] = num->buf[6];

	for (i=0; i< 8; i++)
		num->buf[i] = temp[i];

	u0 = u1 = u2 = u3 = num->ui4[0];
	u4 = num->ui4[1];

	u0 &= 0x80000000;	/* sign bit */
	u1 &= 0x7f800000;	/* exponent */

	if (u1 != 0) {
		u1 >>= 23;
		u1 -= 129;
		u1 += 1023;
		u1 <<= 20;
	}

	u2 &= 0x007fffff;	/* high fraction bits */
	u2 >>= 3;
	u3 &= 0x00000007;	/* low fraction bits */
	u3 <<= 29;
	u4 >>=3;

	num->ui4[0] = (u0|u1|u2);
	num->ui4[1] = (u3|u4);
}

/* shortshuf_ converts VAX 16 bit short integers to Sun shorts */

shortshuf_(num)
short *num;
{
	short left, right;
	
	right = left = *num;
	right >>= 8;
	right &= 0377;
	left <<= 8;
	*num = left|right;
}	

/* intshuf_ changes VAX 32 bit integers to Sun integers */

intshuf_(num)
int *num;
{
	int b1,b2,b3,b4;
	
	b1 = b2 = b3 = b4 = *num;
	b1 >>= 24;
	b2 <<= 8;
	b3 >>= 8;
	b4 <<= 24;

	b2 &= 077600000;
	b3 &= 0177400;
	b1 &= 0377;

	*num = b1|b2|b3|b4;
}	

/* floatshuf_ changes VAX 32 bit floats to Sun floats */

floatshuf_(num)

union {
	float r4;
	unsigned ui4;
} *num;

{
	unsigned b1,b2,b3,b4;

	b1 = b2 = b3 = b4 = num->ui4;
	b1 >>= 8;
	b2 <<= 8;
	b3 >>= 8;
	b4 <<= 8;

	b1 &= 077600000;
	b2 &= 037700000000;
	if (b2 != 0) {
		if (b2 >= 2) {
			b2 >>= 23;
			b2 -= 2;
			b2 <<= 23;
		} else {
			fprintf(stderr,"\nWarning: real*4 number too small");
		}
	}
	b3 &= 0377;
	b4 &= 0177400;

	num->ui4 = (b1|b2|b3|b4);
}	

/* doubleshuf_ changes VAX 64 bit doubles to Sun doubles */

doubleshuf_(num)

union {
	double   r8;
	unsigned ui4[2];
	char     buf[8];
} *num;

{
	char	temp[8];
	unsigned u0, u1, u2, u3, u4;
	int i;

	temp[0] = num->buf[1];
	temp[1] = num->buf[0];
	temp[2] = num->buf[3];
	temp[3] = num->buf[2];
	temp[4] = num->buf[5];
	temp[5] = num->buf[4];
	temp[6] = num->buf[7];
	temp[7] = num->buf[6];

	for (i=0; i< 8; i++)
		num->buf[i] = temp[i];

	u0 = u1 = u2 = u3 = num->ui4[0];
	u4 = num->ui4[1];

	u0 &= 0x80000000;	/* sign bit */
	u1 &= 0x7f800000;	/* exponent */

	if (u1 != 0) {
		u1 >>= 23;
		u1 -= 129;
		u1 += 1023;
		u1 <<= 20;
	}

	u2 &= 0x007fffff;	/* high fraction bits */
	u2 >>= 3;
	u3 &= 0x00000007;	/* low fraction bits */
	u3 <<= 29;
	u4 >>=3;

	num->ui4[0] = (u0|u1|u2);
	num->ui4[1] = (u3|u4);
}
